<?php
// APPS/PanelIconos.php
// Módulo: Panel de Iconos (presentación tipo panel de control)
// Reglas: TODO el PHP al principio. No incluir <html><head><body> (lo pone el padre).

session_start();

// ----------------------
// CONFIGURACIÓN (fácil de ajustar)
// ----------------------
$RESTRINGIR_A_EXT = ['php'];            // extensiones a listar
$CARPETA_APPS     = __DIR__ . '/';      // ruta física de los módulos (APPS/)
$EXCLUIR_FILES    = ['PanelIconos.php']; // archivos a excluir (este mismo)
$ICON_SIZE        = 64;                 // tamaño en px del icono SVG
$COLUMNS_SM       = 2;                  // columnas en pantallas pequeñas
$COLUMNS_MD       = 3;                  // columnas en pantallas medianas
$COLUMNS_LG       = 4;                  // columnas en pantallas grandes
$POPUP_WIDTH      = 1000;               // ancho del popup
$POPUP_HEIGHT     = 700;                // alto del popup

// ----------------------
// SEGURIDAD: Validar sesión
// ----------------------
if (!isset($_SESSION['Acceso']) || $_SESSION['Acceso'] !== 'AccesoSiqueSi') {
    // Retornar HTML mínimo con mensaje (el padre lo inyecta en la plantilla)
    echo '<div class="alert alert-danger">Acceso denegado. Inicia sesión.</div>';
    return;
}

// ----------------------
// FUNCIONES AUXILIARES
// ----------------------
/**
 * Extrae una descripción corta del archivo (.php). Busca la primera línea comentada
 * con formato: // DESC: Texto... o /* DESC: ... * /
 */
function obtener_descripcion_corta($rutaArchivo) {
    $fp = @fopen($rutaArchivo, 'r');
    if (!$fp) return '';
    $desc = '';
    $i = 0;
    while (!feof($fp) && $i < 30) {
        $line = trim(fgets($fp));
        if (preg_match('#^(//|/\*|\#)\s*DESC\s*[:\-]\s*(.+)$#i', $line, $m)) {
            $desc = trim($m[2]);
            break;
        }
        // también soporta comentarios en una línea: /* DESC: ... */
        if (preg_match('#/\*\s*DESC\s*[:\-]\s*(.+)\s*\*/#i', $line, $m2)) {
            $desc = trim($m2[1]);
            break;
        }
        $i++;
    }
    fclose($fp);
    return $desc;
}

/**
 * Determina el título amigable para el archivo (sin extensión y con espacios)
 */
function titulo_amigable($nombreArchivo) {
    $sinExt = preg_replace('/\.[^.]+$/', '', $nombreArchivo);
    // Reemplaza guiones/underscores por espacios y capitaliza cada palabra
    $sinExt = preg_replace('/[-_]+/', ' ', $sinExt);
    return ucwords($sinExt);
}

// ----------------------
// LISTAR ARCHIVOS EN APPS/
// ----------------------
$items = [];
$dir = @opendir($CARPETA_APPS);
if ($dir) {
    while (($file = readdir($dir)) !== false) {
        if (in_array($file, ['.', '..'])) continue;
        if (in_array($file, $EXCLUIR_FILES)) continue;
        $full = $CARPETA_APPS . $file;
        if (!is_file($full)) continue;
        $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
        if (!in_array($ext, $RESTRINGIR_A_EXT)) continue;
        // Guardar item
        $items[] = [
            'file' => $file,
            'title' => titulo_amigable($file),
            'desc' => obtener_descripcion_corta($full),
        ];
    }
    closedir($dir);
}

// Ordenar por título
usort($items, function($a,$b){ return strcasecmp($a['title'],$b['title']); });

// ----------------------
// Salida del HTML del módulo (Bootstrap)
// ----------------------
?>
<!-- Scoped styles: usa clases específicas para evitar colisiones -->
<style>
/* Scoped al módulo - prefijo .panel-iconos- para evitar romper tema */
.panel-iconos-grid { margin-top: 8px; }
.panel-icono-card {
    transition: transform .12s ease, box-shadow .12s ease;
    cursor: pointer;
}
.panel-icono-card:hover {
    transform: translateY(-6px);
    box-shadow: 0 10px 20px rgba(0,0,0,0.08);
}
.panel-icono-card .card-body { padding: .75rem; }
.panel-icono-icon {
    width: <?php echo intval($ICON_SIZE); ?>px;
    height: <?php echo intval($ICON_SIZE); ?>px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    border-radius: .5rem;
    background: rgba(0,0,0,0.03);
}
.panel-icono-title { font-size: .95rem; font-weight:600; margin-top: .5rem; }
.panel-icono-desc { font-size: .82rem; color: #666; }
.panel-iconos-empty { padding: 2rem; text-align:center; color:#777; }
.small-muted { font-size:.78rem; color:#999; }
</style>

<div class="card">
  <div class="card-body">
    <div class="d-flex justify-content-between align-items-start mb-2">
      <div>
        <h5 class="mb-0">Panel de Módulos</h5>
        <div class="small-muted">Haz clic en un icono para abrir el módulo en una ventana hija minimal.</div>
      </div>
      <div>
        <button class="btn btn-sm btn-outline-secondary" onclick="abrirTodoEnPestanias()">Abrir todo (pestañas)</button>
        <button class="btn btn-sm btn-outline-primary" onclick="abrirTodoEnPopups()">Abrir todo (popups)</button>
      </div>
    </div>

    <?php if (empty($items)): ?>
      <div class="panel-iconos-empty">No se encontraron módulos en la carpeta <code>APPS/</code>.</div>
    <?php else: ?>
      <div class="row panel-iconos-grid">
        <?php foreach ($items as $it): 
            // nombre sin extension para pasar por GET (ej: index.php?A=NombreModulo)
            $sinExt = preg_replace('/\.[^.]+$/', '', $it['file']);
            // título/desc
            $title = htmlspecialchars($it['title'], ENT_QUOTES, 'UTF-8');
            $desc  = htmlspecialchars($it['desc'] ?: 'Abrir módulo ' . $title, ENT_QUOTES, 'UTF-8');
        ?>
          <div class="col-6 col-md-<?php echo (12 / max(1,$COLUMNS_MD)); ?> col-lg-<?php echo (12 / max(1,$COLUMNS_LG)); ?> mb-3">
            <div class="card panel-icono-card" role="button"
                 onclick="abrirModuloMinimal('<?php echo rawurlencode($sinExt); ?>', '<?php echo addslashes($title); ?>')">
              <div class="card-body d-flex align-items-center">
                <div class="panel-icono-icon me-3" aria-hidden="true">
                  <!-- Icono SVG genérico; puedes personalizar por archivo leyendo una convención -->
                  <svg width="<?php echo $ICON_SIZE;?>" height="<?php echo $ICON_SIZE;?>" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" >
                    <rect x="2" y="3" width="20" height="14" rx="2" stroke="currentColor" stroke-width="1.2" fill="none"/>
                    <rect x="6" y="7" width="4" height="4" rx="0.5" stroke="currentColor" stroke-width="1.2" fill="none"/>
                    <path d="M7 16h10" stroke="currentColor" stroke-width="1.2"/>
                  </svg>
                </div>
                <div>
                  <div class="panel-icono-title"><?php echo $title; ?></div>
                  <div class="panel-icono-desc"><?php echo $desc; ?></div>
                </div>
              </div>
              <div class="card-footer small text-muted">Archivo: <code><?php echo htmlspecialchars($it['file']); ?></code></div>
            </div>
          </div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<script>
// Configuración JS (ajustable desde arriba en PHP si lo deseas)
const POPUP_WIDTH = <?php echo intval($POPUP_WIDTH); ?>;
const POPUP_HEIGHT = <?php echo intval($POPUP_HEIGHT); ?>;

/**
 * Construye la URL absoluta para abrir el módulo usando el loader del sistema (index.php?A=Nombre)
 * Esto asume que el módulo se carga por la ruta actual que contiene index.php (el padre).
 */
function urlModulo(nombreSinExt) {
  // usamos window.location to preserve el path del padre (index.php)
  const base = window.location.origin + window.location.pathname;
  // Si la URL actual ya tiene ?A=..., limpiar querystring
  const clean = base.split('?')[0];
  return clean + '?A=' + encodeURIComponent(nombreSinExt);
}

/**
 * Abre un popup minimal (sin barra de herramientas). Los navegadores modernos pueden ignorar
 * algunos flags, pero esta es la forma estándar.
 */
function abrirPopupMinimal(url, titulo, w = POPUP_WIDTH, h = POPUP_HEIGHT) {
  const left = Math.max(0, (screen.width - w) / 2);
  const top  = Math.max(0, (screen.height - h) / 2);
  const features = [
    'toolbar=no',
    'location=no',
    'status=no',
    'menubar=no',
    'scrollbars=yes',
    'resizable=yes',
    'width=' + w,
    'height=' + h,
    'left=' + left,
    'top=' + top
  ].join(',');
  // nombre de ventana único por título
  const win = window.open(url, 'mod_' + titulo.replace(/\s+/g,'_'), features);
  if (!win) {
    alert('El navegador bloqueó la apertura de ventanas emergentes. Permite popups o usa "Abrir todo (pestañas)".');
    return null;
  }
  win.focus();
  return win;
}

/**
 * Abrir el módulo en una ventana hija minimal (usa index.php?A=NombreModulo)
 */
function abrirModuloMinimal(nombreSinExt, titulo) {
  const url = urlModulo(decodeURIComponent(nombreSinExt));
  abrirPopupMinimal(url, titulo);
}

/**
 * Abrir todos los módulos en pestañas (útil si el usuario quiere trabajar en varias)
 */
function abrirTodoEnPestanias() {
  const cards = document.querySelectorAll('.panel-icono-card');
  cards.forEach((c, idx) => {
    // tomar onclick que contiene la llamada con nombre codificado
    const onclick = c.getAttribute('onclick') || '';
    // extraer el primer argumento entre comillas simples
    const m = onclick.match(/abrirModuloMinimal\('([^']+)'/);
    if (m) {
      const nombre = decodeURIComponent(m[1]);
      const url = urlModulo(nombre);
      // abrir en _blank para forzar nueva pestaña (si el navegador lo permite)
      window.open(url, '_blank');
    }
  });
}

/**
 * Abrir todo en popups (una ventana por módulo)
 */
function abrirTodoEnPopups() {
  const cards = document.querySelectorAll('.panel-icono-card');
  cards.forEach((c, idx) => {
    const onclick = c.getAttribute('onclick') || '';
    const m = onclick.match(/abrirModuloMinimal\('([^']+)'/);
    if (m) {
      const nombre = decodeURIComponent(m[1]);
      const titulo = c.querySelector('.panel-icono-title') ? c.querySelector('.panel-icono-title').innerText : nombre;
      abrirPopupMinimal(urlModulo(nombre), titulo, POPUP_WIDTH, POPUP_HEIGHT);
    }
  });
}
</script>
