<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<title>Rastreador de Hábitos Circular Avanzado</title>

<style>
    body {
        font-family: 'Arial', sans-serif;
        display: flex;
        flex-direction: column;
        justify-content: center;
        align-items: center;
        min-height: 100vh;
        background: #f4f4f4;
        margin: 0;
    }

    .container {
        text-align: center;
        background: white;
        padding: 25px 40px;
        border-radius: 15px;
        box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        width: 90%;
        max-width: 700px;
    }

    /* Controles y Botones */
    .controls {
        display: flex;
        gap: 10px;
        justify-content: center;
        margin-bottom: 15px;
    }
    .controls button {
        padding: 10px 15px; 
        background: #3498db; 
        color: white; 
        border: none; 
        border-radius: 5px;
        cursor: pointer;
        font-weight: bold;
    }
    .controls button:hover {
        background: #2980b9;
    }

    /* Slider de meses */
    .month-slider-container {
        margin: 15px 0;
        text-align: center;
    }
    .month-slider-container label {
        display: block;
        margin-bottom: 5px;
        font-weight: bold;
    }
    .month-slider-container input[type="range"] {
        width: 80%;
        max-width: 400px;
        margin: 0 auto;
    }
    
    .habit-list {
        margin-top: 10px;
        text-align: left;
        font-size: 14px;
        max-height: 150px; 
        overflow-y: auto; 
        padding-right: 10px; 
    }

    .habit-item {
        display: flex;
        justify-content: space-between;
        align-items: center; 
        padding: 3px 5px; 
        border-radius: 5px;
        transition: background-color 0.2s ease;
        cursor: pointer;
    }

    .delete-btn {
        background: #e74c3c;
        margin-left: 10px;
        padding: 4px 8px;
        line-height: 1;
        font-weight: normal;
    }
    .delete-btn:hover { background: #c0392b; }

    /* --- Highlighting Styles --- */
    .habit-item.highlighted {
        background-color: #ffe0b2; 
        font-weight: bold;
        box-shadow: 0 0 5px rgba(255, 165, 0, 0.5);
    }
    .habit-label.highlighted {
        fill: #ff8c00; 
    }
    .highlighted-path {
        stroke: #ff8c00 !important; 
        stroke-width: 2px !important;
        opacity: 1 !important;
    }
    /* SVG Styles */
    svg { 
        max-width: 600px; 
        margin-top: 20px; 
        transition: transform 0.8s ease-in-out; 
    }
    path { 
        transition: fill 0.2s ease, opacity 0.2s, stroke 0.2s ease; 
        stroke: #333; 
        stroke-width: 0.5px; 
        cursor: pointer; 
    }
    path:hover { 
        opacity: 0.85; 
    }
    .day-number { 
        font-size: 10px; 
        fill: #333; 
        text-anchor: middle; 
        dominant-baseline: central;
    }
    .habit-label { 
        font-size: 12px; 
        fill: #333; 
        text-anchor: end;
        cursor: pointer; 
        font-weight: bold;
        transition: fill 0.2s ease;
    }
    .habit-label:hover {
        fill: #000;
    }
    .view-day-marker { 
        fill: none; 
        stroke: #e74c3c; 
        stroke-width: 3px; 
        opacity: 0.7;
    }
    .completed {
        cursor: default !important; 
    }

    /* --- Estilos del Modal --- */
    .modal {
        display: none; 
        position: fixed; 
        z-index: 100; 
        left: 0;
        top: 0;
        width: 100%; 
        height: 100%;
        overflow: auto; 
        background-color: rgba(0,0,0,0.4); 
        padding-top: 60px;
    }
    .modal-content {
        background-color: #fefefee;
        margin: 5% auto; 
        padding: 20px;
        border: 1px solid #888;
        width: 80%; 
        max-width: 300px;
        border-radius: 10px;
        text-align: left;
        box-shadow: 0 5px 15px rgba(0,0,0,0.3);
        animation-name: animatetop;
        animation-duration: 0.4s
    }

    @keyframes animatetop {
        from {top: -300px; opacity: 0}
        to {top: 0; opacity: 1}
    }

    .close-btn {
        color: #aaa;
        float: right;
        font-size: 28px;
        font-weight: bold;
    }
    .close-btn:hover,
    .close-btn:focus {
        color: #000;
        text-decoration: none;
        cursor: pointer;
    }
</style>

</head>
<body>

<div class="container">
    <h2>🎯 Rastreador de Hábitos</h2>

    <div class="controls">
        <button onclick="openModal()">➕ Añadir Hábito</button>
        <button id="saveBtn">💾 Guardar Progreso</button>
    </div>

    <div class="month-slider-container">
        <label for="monthSlider">Mes en Vista: <span id="currentMonthDisplay"></span></label>
        <input type="range" id="monthSlider" min="-12" max="0" value="0">
    </div>

    <div class="habit-list" id="habitList"></div>

    <div id="chart-container"></div>

    <p style="font-size:12px;color:#777;">
        Hoy es <span id="trueCurrentDateDisplay"></span>. 
        Progreso guardado automáticamente al marcar/desmarcar.
    </p>
</div>

<div id="addHabitModal" class="modal">
    <div class="modal-content">
        <span class="close-btn" onclick="closeModal('addHabitModal')">&times;</span>
        <h3>Nuevo Hábito</h3>
        <p>Nombre del Hábito:</p>
        <input type="text" id="modalHabitNameInput" style="width: 90%; padding: 8px; margin-bottom: 15px; border: 1px solid #ccc; border-radius: 5px;">
        <button style="width: 100%;" onclick="submitNewHabit()">Crear Hábito</button>
    </div>
</div>


<script>
// ========================
// CONFIGURACIÓN GLOBAL
// ========================
const STORAGE_KEY_PREFIX = "circularHabitData";
const MAX_HABITOS = 7;

const SVG_SIZE = 500;
const CENTER = SVG_SIZE / 2;
const GAP_ANGULO = 90; // Hueco para los labels
const TRACK_WIDTH = 25;
const BASE_RADIUS = 80;

// ************************************************************
// NUEVA VARIABLE: AJUSTA ESTE VALOR PARA CAMBIAR LA ROTACIÓN INICIAL
// 0: Centra el día actual en la parte superior (90 grados)
// Puedes usar valores positivos o negativos para rotar
const CUSTOM_START_ROTATION = 112; // Ejemplo: 45 para rotar 45 grados adicionales
// ************************************************************

// Estado
let selectedHabitId = null; // ID del hábito actualmente seleccionado/resaltado

// Fechas
const TRUE_CURRENT_DATE = new Date();
const monthNames = ["Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio", "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"];

document.getElementById("trueCurrentDateDisplay").textContent = TRUE_CURRENT_DATE.toLocaleDateString('es-ES', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' });

let CURRENT_DATE_VIEW = new Date(TRUE_CURRENT_DATE); 
let CURRENT_MONTH_OFFSET = 0; 
let habits = []; 

let DIAS_MES_VIEW = 0; 
let VIEW_CURRENT_DAY = 0; 
let ROTATION_OFFSET = 0; 


// ========================
// UTILIDADES Y FECHAS
// ========================
function getRandomColor() {
    return `hsl(${Math.random() * 360}, 70%, 50%)`;
}

function getDaysInMonth(year, month) {
    return new Date(year, month + 1, 0).getDate();
}

// --- Lógica del Slider de Meses ---
const monthSlider = document.getElementById('monthSlider');
monthSlider.addEventListener('input', (event) => {
    CURRENT_MONTH_OFFSET = parseInt(event.target.value);
    highlightSelection(null); // Limpiar selección al cambiar de mes
    updateViewDate();
    renderEverything();
});

function getStorageKeyForMonth(date) {
    return `${STORAGE_KEY_PREFIX}-${date.getFullYear()}-${date.getMonth()}`;
}

function updateViewDate() {
    // 1. Calcular la fecha de vista
    CURRENT_DATE_VIEW = new Date(TRUE_CURRENT_DATE);
    CURRENT_DATE_VIEW.setMonth(TRUE_CURRENT_DATE.getMonth() + CURRENT_MONTH_OFFSET);
    
    DIAS_MES_VIEW = getDaysInMonth(CURRENT_DATE_VIEW.getFullYear(), CURRENT_DATE_VIEW.getMonth());

    // 2. Determinar el "día actual" de la vista (hasta dónde podemos marcar)
    if (CURRENT_MONTH_OFFSET === 0) { 
        VIEW_CURRENT_DAY = TRUE_CURRENT_DATE.getDate();
    } else if (CURRENT_MONTH_OFFSET < 0) { 
        VIEW_CURRENT_DAY = DIAS_MES_VIEW; 
    } else { 
        VIEW_CURRENT_DAY = 0; 
    }
    
    document.getElementById('currentMonthDisplay').textContent = `${monthNames[CURRENT_DATE_VIEW.getMonth()]} ${CURRENT_DATE_VIEW.getFullYear()}`;

    // 3. Recalcular offset de rotación para centrar el día de vista a 90 grados (arriba)
    
    const totalAngle = 360 - GAP_ANGULO;
    const anglePerDay = totalAngle / DIAS_MES_VIEW;
    const START_ANGLE_RADIAL = 90 - (GAP_ANGULO / 2); 

    let dayToCenter = (VIEW_CURRENT_DAY > 0) ? VIEW_CURRENT_DAY : 1;
    
    const CURRENT_DAY_ARC_ANGLE = START_ANGLE_RADIAL + (anglePerDay * (dayToCenter - 1)) + (anglePerDay / 2);
    
    ROTATION_OFFSET = 90 - CURRENT_DAY_ARC_ANGLE; 

    // Añadir la rotación personalizada
    ROTATION_OFFSET += CUSTOM_START_ROTATION;
}


// ========================
// PERSISTENCIA
// ========================
function saveData() {
    const key = getStorageKeyForMonth(CURRENT_DATE_VIEW);
    localStorage.setItem(key, JSON.stringify(habits));
}

function loadData() {
    const key = getStorageKeyForMonth(CURRENT_DATE_VIEW);
    const storedData = localStorage.getItem(key);
    return storedData ? JSON.parse(storedData) : [];
}


// ========================
// HABITS CRUD
// ========================
function addHabit() {
    const input = document.getElementById("modalHabitNameInput"); 
    const name = input.value.trim();

    if (!name) {
        alert("Escribe un nombre para el hábito.");
        return;
    }
    
    if (habits.length >= MAX_HABITOS) {
        alert(`Solo puedes tener un máximo de ${MAX_HABITOS} hábitos activos en este mes.`);
        return;
    }

    habits.push({
        id: Date.now(),
        nombre: name,
        color: getRandomColor(),
        dias: []
    });

    input.value = ""; 
    saveData();
    renderEverything();
    closeModal('addHabitModal'); 
}

function deleteHabit(id) {
    habits = habits.filter(h => h.id !== id);
    saveData();
    renderEverything();
}


// ========================
// SVG GEOMETRÍA
// ========================
function polar(cx, cy, r, ang) {
    const rad = (ang - 90) * Math.PI / 180; 
    return { x: cx + r * Math.cos(rad), y: cy + r * Math.sin(rad) };
}

function arcPath(cx, cy, r1, r2, a1, a2) {
    var startOuter = polar(cx, cy, r2, a2);
    var endOuter = polar(cx, cy, r2, a1);
    var startInner = polar(cx, cy, r1, a2);
    var endInner = polar(cx, cy, r1, a1);

    var largeArcFlag = a2 - a1 <= 180 ? "0" : "1";

    var d = [
        "M", startOuter.x, startOuter.y, 
        "A", r2, r2, 0, largeArcFlag, 0, endOuter.x, endOuter.y, 
        "L", endInner.x, endInner.y, 
        "A", r1, r1, 0, largeArcFlag, 1, startInner.x, startInner.y, 
        "Z"
    ].join(" ");

    return d;
}


// ========================
// HIGHLIGHTING Y TOGGLE DÍA
// ========================
function highlightSelection(habitId) {
    selectedHabitId = habitId;

    // 1. Limpiar todos los highlights
    document.querySelectorAll('.habit-item').forEach(el => el.classList.remove('highlighted'));
    document.querySelectorAll('.habit-label').forEach(el => el.classList.remove('highlighted'));
    document.querySelectorAll('path').forEach(el => el.classList.remove('highlighted-path'));

    if (habitId) {
        // 2. Resaltar en la lista y el label
        const habitItem = document.querySelector(`.habit-item[data-habit-id="${habitId}"]`);
        const labelText = document.querySelector(`.habit-label[data-habit-id="${habitId}"]`);
        
        if (habitItem) habitItem.classList.add('highlighted');
        if (labelText) labelText.classList.add('highlighted');
        
        // 3. Resaltar en el SVG (todo el anillo del hábito)
        document.querySelectorAll(`path[data-habit-id="${habitId}"]`).forEach(pathEl => {
            pathEl.classList.add('highlighted-path');
        });
    }
}

function toggleDay(habitId, day, pathEl, force = false) {
    // Si no es el día de vista actual, sólo se permite marcar si no está ya completado
    if (day !== VIEW_CURRENT_DAY && !force) {
        // Solo para marcar días históricos (anteriores a VIEW_CURRENT_DAY)
        if (day > VIEW_CURRENT_DAY) {
            alert("No puedes marcar días futuros para el mes en vista.");
            return;
        }
    }
    
    const habit = habits.find(h => h.id === habitId);
    if (!habit) return;

    const idx = habit.dias.indexOf(day);

    if (idx >= 0) {
        // Desmarcar (Solo permitido si day es el VIEW_CURRENT_DAY o si se permite la desmarcación forzada)
        if (day === VIEW_CURRENT_DAY) {
            habit.dias.splice(idx, 1);
            pathEl.setAttribute("fill", "none");
            pathEl.classList.remove("completed");
        } else {
             // Bloquear desmarcación de días históricos
            alert("No se permite desmarcar días completados (solo el día actual puede alternarse).");
            return;
        }
    } else {
        // Marcar
        habit.dias.push(day);
        pathEl.setAttribute("fill", habit.color);
        pathEl.classList.add("completed");
    }

    saveData();
}

function toggleCurrentDay(habitId) {
    if (VIEW_CURRENT_DAY <= 0) {
        alert("El mes actual no permite modificar días.");
        return;
    }

    const pathEl = document.querySelector(`path[data-habit-id="${habitId}"][data-day="${VIEW_CURRENT_DAY}"]`);
    
    if (pathEl) {
        toggleDay(habitId, VIEW_CURRENT_DAY, pathEl, true); 
        highlightSelection(habitId); 
    }
}


// ========================
// RENDER HABIT LIST
// ========================
function renderHabitList() {
    const list = document.getElementById("habitList");
    list.innerHTML = habits
        .map(h => `
            <div class="habit-item ${h.id === selectedHabitId ? 'highlighted' : ''}" data-habit-id="${h.id}" onclick="toggleCurrentDay(${h.id}); highlightSelection(${h.id});">
                <span style="color: ${h.color}; margin-right: 5px;">⬤</span>
                ${h.nombre}
                <button class="delete-btn" onclick="event.stopPropagation(); deleteHabit(${h.id});">✖</button>
            </div>
        `).join("");
}


// ========================
// RENDER SVG CHART
// ========================
function renderChart() {
    habits = loadData(); 
    const container = document.getElementById("chart-container");
    container.innerHTML = "";

    const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
    svg.setAttribute("width", SVG_SIZE);
    svg.setAttribute("height", SVG_SIZE);
    svg.setAttribute("viewBox", `0 0 ${SVG_SIZE} ${SVG_SIZE}`);
    
    const gRotator = document.createElementNS("http://www.w3.org/2000/svg", "g");
    gRotator.setAttribute("transform", `rotate(${ROTATION_OFFSET}, ${CENTER}, ${CENTER})`);
    
    const gLabels = document.createElementNS("http://www.w3.org/2000/svg", "g");

    svg.appendChild(gRotator);
    svg.appendChild(gLabels);


    const totalAngle = 360 - GAP_ANGULO;
    const anglePerDay = totalAngle / DIAS_MES_VIEW;
    const START_ANGLE_RADIAL = 90 - (GAP_ANGULO / 2); 

    habits.forEach((habit, i) => {
        const inner = BASE_RADIUS + i * TRACK_WIDTH;
        const outer = inner + TRACK_WIDTH;

        // 1. DIBUJAR LABELS (NO ROTAN)
        const labelXStart = CENTER - 120; 
        const labelXEnd = CENTER - (outer + 10); 
        const labelY = CENTER - (inner + outer) / 2; 
        const isHighlighted = habit.id === selectedHabitId;

        let line = document.createElementNS("http://www.w3.org/2000/svg", "line");
        line.setAttribute("x1", labelXStart + 5);
        line.setAttribute("y1", labelY);
        line.setAttribute("x2", labelXEnd);
        line.setAttribute("y2", labelY);
        line.setAttribute("stroke", isHighlighted ? '#ff8c00' : "#555");
        line.setAttribute("stroke-width", "0.5");
        gLabels.appendChild(line);

        let labelText = document.createElementNS("http://www.w3.org/2000/svg", "text");
        labelText.setAttribute("x", labelXStart);
        labelText.setAttribute("y", labelY + 4); 
        labelText.setAttribute("class", `habit-label ${isHighlighted ? 'highlighted' : ''}`);
        labelText.setAttribute("data-habit-id", habit.id);
        labelText.textContent = habit.nombre;
        
        // El clic en el label ahora alterna el día actual y resalta
        if (VIEW_CURRENT_DAY > 0) {
            labelText.addEventListener('click', (e) => {
                toggleCurrentDay(habit.id);
            }); 
        } else {
            labelText.style.cursor = "default";
            labelText.style.opacity = 0.5;
        }
        gLabels.appendChild(labelText); 

        // 2. DIBUJAR SEGMENTOS (ROTAN)
        for (let d = 1; d <= DIAS_MES_VIEW; d++) {
            const a1 = START_ANGLE_RADIAL + (d - 1) * anglePerDay;
            const a2 = a1 + anglePerDay;

            const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
            path.setAttribute("d", arcPath(CENTER, CENTER, inner, outer, a1, a2));

            const done = habit.dias.includes(d);
            path.setAttribute("fill", done ? habit.color : "none");

            path.dataset.habitId = habit.id;
            path.dataset.day = d;
            
            if (isHighlighted) {
                 path.classList.add('highlighted-path');
            }

            const isClickable = d <= VIEW_CURRENT_DAY && !done;
            
            if (isClickable) {
                path.style.opacity = "1";
            } else {
                path.style.opacity = done ? "1" : "0.3"; 
                if (done) path.classList.add("completed");
            }
            gRotator.appendChild(path);
        }
    });

    // 3. DIBUJAR NÚMEROS Y MARCADOR (ROTAN)
    const textRadius = BASE_RADIUS + habits.length * TRACK_WIDTH + 12;
    const highlightRadius = textRadius + 10;

    for (let d = 1; d <= DIAS_MES_VIEW; d++) {
        const ang = START_ANGLE_RADIAL + (d - 1) * anglePerDay + anglePerDay / 2;

        const p = polar(CENTER, CENTER, textRadius, ang);
        const mark = polar(CENTER, CENTER, highlightRadius, ang);

        // Marcar el "Día de Vista Actual"
        if (d === VIEW_CURRENT_DAY && VIEW_CURRENT_DAY > 0) {
            const circle = document.createElementNS("http://www.w3.org/2000/svg", "circle");
            circle.setAttribute("cx", mark.x);
            circle.setAttribute("cy", mark.y);
            circle.setAttribute("r", 13);
            circle.setAttribute("class", "view-day-marker");
            gRotator.appendChild(circle);
        }

        const t = document.createElementNS("http://www.w3.org/2000/svg", "text");
        t.setAttribute("x", p.x);
        t.setAttribute("y", p.y);
        t.classList.add("day-number");
        t.setAttribute("transform", `rotate(${90}, ${p.x}, ${p.y})`); 
        t.textContent = d;
        gRotator.appendChild(t);
    }

    container.appendChild(svg);

    // EVENTO DELEGADO EN EL GRUPO ROTATORIO
    gRotator.addEventListener("click", e => {
        const el = e.target;
        if (el.tagName !== 'path') return;
        
        const habitId = Number(el.dataset.habitId);
        const day = Number(el.dataset.day);

        highlightSelection(habitId); // Resaltar el hábito

        // Si el día NO es el VIEW_CURRENT_DAY, solo permitimos marcar
        if (day !== VIEW_CURRENT_DAY) {
            if (!el.classList.contains("completed") && day <= VIEW_CURRENT_DAY) {
                toggleDay(habitId, day, el, false); // No forzar toggle (solo marcar)
            }
        } else {
             // Si el día ES el VIEW_CURRENT_DAY, permitimos alternar (marcar/desmarcar)
            toggleDay(habitId, day, el, true);
        }
    });
}


// ========================
// MODAL LOGIC
// ========================
function openModal(modalId = 'addHabitModal') {
    document.getElementById(modalId).style.display = 'block';
    if (modalId === 'addHabitModal') {
        document.getElementById('modalHabitNameInput').focus();
    }
}

function closeModal(modalId = 'addHabitModal') {
    document.getElementById(modalId).style.display = 'none';
}

window.onclick = function(event) {
    if (event.target.classList.contains('modal')) {
        closeModal(event.target.id);
    }
}

function submitNewHabit() {
    addHabit();
}


// ========================
// RENDER TODO
// ========================
function renderEverything() {
    renderHabitList();
    renderChart();
}

document.getElementById("saveBtn").onclick = saveData;

// Inicialización
updateViewDate();
renderEverything();
</script>

</body>
</html>